//******************************************************************************
//Name          : PoleZero.cs
//Description   : Implementation to experience the beauty and the pitfalls of the
//                pole-zero-plane
//Author        : Dr. Oliver Kluge, 90411 Nuernberg, Germany
//Copyright(C)  : Oliver Kluge 2014 
//------------------------------------------------------------------------------
//******************************************************************************

using System;
using System.Globalization;
using Display;


struct complex
{
    public double re;
    public double im;
}


class root
{
    public int number;
    public complex[] z;

    public root(int dim)
    {
        number = 0;
        z = new complex[dim];
    }
}


class PoleZero
{
    const int samplerate = 128;                         // Abtastwerte je Periode
    const double PI      =   3.141592653589793;
    const int maxPole    =  10;
    const int maxZero    =  10;

    public root Zero = new root(maxZero);
    public root Pole = new root(maxPole);

    public double[] AS = new double[samplerate];        // Amplitudenspektrum
    public double[] PS = new double[samplerate];        // Phasenspektrum
    public double[] arg = new double[samplerate];

    public void FrequencyResponse()
    {
        double[] Lo   = new double[Zero.number];        // Nullstellen Differenzzeiger
        double[] Lx   = new double[Pole.number];        // Polstellen Differenzzeiger
        double Lre    = 0;                              // Zeiger-Realteil
        double Lim    = 0;                              // Zeiger-Imaginrteil
        double Po     = 0;                              // Produkt der Nullstellenzeiger
        double Px     = 0;                              // Produkt der Polstellenzeiger
        double fa     = samplerate;                     // Abtastfrequenz
        double fstart = 0.0;                            // Startfrequenz
        double fstop  = samplerate/2;                   // Stopfrequenz
        double deltaf = (fstop-fstart)/(samplerate-1);  // Frequenznderung
        double freq   = fstart;                         // variable Frequenz

        for (int i=0; i<samplerate; i++)
        {
            Po    = 1;
            Px    = 1;
            PS[i] = 0;

            for (int k=0; k<Zero.number; k++)
            {
                // Differenzzeiger aller Nullstellen berechnen
                Lre   = Math.Cos(2*PI*freq/fa)-Zero.z[k].re;
                Lim   = Math.Sin(2*PI*freq/fa)-Zero.z[k].im;
                Lo[k] = Math.Sqrt(Lre*Lre + Lim*Lim);

                // Multiplikation der Zeigerbetrge
                Po *= Lo[k];
                // Summation der Zeigerwinkel
                PS[i] += Math.Atan2(Lim, Lre + 1E-10);
            }

            for (int k=0; k<Pole.number; k++)
            {
                // Differenzzeiger aller Polstellen berechnen
                Lre   = Math.Cos(2*PI*freq/fa)-Pole.z[k].re;
                Lim   = Math.Sin(2*PI*freq/fa)-Pole.z[k].im;
                Lx[k] = Math.Sqrt(Lre*Lre + Lim*Lim);

                // Multiplikation der Zeigerbetrge
                Px *= Lx[k];
                // Summation der Zeigerwinkel
                PS[i] -= Math.Atan2(Lim, Lre + 1E-10);
            }

            AS[i] = Po/Px;
            arg[i] = freq/fa;
			
            freq += deltaf;
        }
    }
}


class MainApp
{
    [STAThread]
    static void Main()
    {
        string userinput = "";
        PoleZero mySystem = new PoleZero();

        Console.Write("Calculate system's frequency response from its pole/zero-plane\n\n");
        Console.Write("Bitte Anzahl der Nullstellen eingeben: ");
        userinput = Console.ReadLine();
        mySystem.Zero.number = Convert.ToInt16(userinput);
        for (int i=0; i<mySystem.Zero.number; i++)
        {
            Console.Write("\n{0, 3:d}. Nullstelle:\n", i+1);
            Console.Write("  real = ");
            userinput = Console.ReadLine();
            mySystem.Zero.z[i].re = Convert.ToDouble(userinput, CultureInfo.InvariantCulture);
            Console.Write("  imag = ");
            userinput = Console.ReadLine();
            mySystem.Zero.z[i].im = Convert.ToDouble(userinput, CultureInfo.InvariantCulture);
        }

        Console.Write("\nBitte Anzahl der Polstellen eingeben: ");
        userinput = Console.ReadLine();
        mySystem.Pole.number = Convert.ToInt16(userinput);
        for (int i=0; i<mySystem.Pole.number; i++)
        {
            Console.Write("\n{0, 3:d}. Polstelle:\n", i+1);
            Console.Write("  real = ");
            userinput = Console.ReadLine();
            mySystem.Pole.z[i].re = Convert.ToDouble(userinput, CultureInfo.InvariantCulture);
            Console.Write("  imag = ");
            userinput = Console.ReadLine();
            mySystem.Pole.z[i].im = Convert.ToDouble(userinput, CultureInfo.InvariantCulture);
        }

        mySystem.FrequencyResponse();

        // create a chart for displaying that data...
        DisplayChart mydisplay1 = new DisplayChart(mySystem.arg, mySystem.AS, "Frequency response", "f/fs", "|H(f)|");
        mydisplay1.Plot();

        DisplayChart mydisplay2 = new DisplayChart(mySystem.arg, mySystem.PS, "Frequency response", "f/fs", "Phi(f)");
        mydisplay2.Plot();
    }
}
